/*
 * Copyright (C) 1998  Junichi Satoh <junichi@astec.co.jp>
 *   All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR(S) ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR(S) BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 *      $Id: $
 */
/*
 *  ATAPI LS-120/ZIP control utility.
 *  This is based on cdcontrol.
 */


/*
 * Compact Disc Control Utility by Serge V. Vakulenko <vak@cronyx.ru>.
 * Based on the non-X based CD player by Jean-Marc Zucconi and
 * Andrey A. Chernov.
 *
 * Fixed and further modified on 5-Sep-1995 by Jukka Ukkonen <jau@funet.fi>.
 *
 * 11-Sep-1995: Jukka A. Ukkonen <jau@funet.fi>
 *              A couple of further fixes to my own earlier "fixes".
 *
 * 18-Sep-1995: Jukka A. Ukkonen <jau@funet.fi>
 *              Added an ability to specify addresses relative to the
 *              beginning of a track. This is in fact a variation of
 *              doing the simple play_msf() call.
 *
 * 11-Oct-1995: Serge V.Vakulenko <vak@cronyx.ru>
 *              New eject algorithm.
 *              Some code style reformatting.
 */

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/cdio.h>
#include <sys/ioctl.h>
#include <machine/ioctl_wfd.h>

#define VERSION "1.0"

#ifndef DEFAULT_WFD_DRIVE
#  define DEFAULT_WFD_DRIVE  "/dev/wfd0c"
#endif

#ifndef DEFAULT_WFD_PARTITION
#  define DEFAULT_WFD_PARTITION  "c"
#endif

#define LS120	0
#define ZIP	1

#define MDT_UNKNOWN     0x00
#define MDT_2DD_UN	0x10
#define MDT_2DD		0x11
#define MDT_2HD_UN	0x20
#define MDT_2HD_12_98	0x22
#define MDT_2HD_12	0x23
#define MDT_2HD_144	0x24
#define MDT_LS120	0x31

#define ZDS_UNKNOWN_STATUS	1
#define ZDS_SPIN_SPEED		2
#define ZDS_SPIN_UP		3
#define ZDS_NO_DISK		4
#define ZDS_AUTO_DOWN		5
#define ZDS_MOTOR_STALL		6

#define CMD_DEBUG       1
#define CMD_EJECT       2
#define CMD_HELP        3
#define CMD_QUIT        7
#define CMD_RESUME      8
#define CMD_RESET       12
#define CMD_SET         13
#define CMD_STATUS	14
#define CMD_PROTECT	100
#define CMD_FORMAT	101

struct cmdtab {
	int command;
	char *name;
	unsigned  min;
	char *args;
} cmdtab[] = {
{ CMD_DEBUG,    "debug",        1, "on | off" },
{ CMD_EJECT,    "eject",        1, "" },
{ CMD_HELP,     "?",            1, 0 },
{ CMD_HELP,     "help",         1, "" },
{ CMD_QUIT,     "quit",         1, "" },
{ CMD_RESET,    "reset",        4, "" },
{ CMD_RESUME,   "resume",       1, "" },
{ CMD_PROTECT,  "protect",      1, "on | rwon | off" },
{ CMD_FORMAT,   "format",       1, "" },
{ CMD_STATUS,	"status",	1, "" },
{ 0, }
};

const char	*wfdname;
int		fd = -1;
int		verbose = 1;

int             status __P((char *arg));
int             open_wfd __P((void));
char            *input __P((int *));
int             run __P((int cmd, char *arg));
char            *parse __P((char *buf, int *cmd));

void help ()
{
	struct cmdtab *c;
	char *s, n;
	int i;

	for (c=cmdtab; c->name; ++c) {
		if (! c->args)
			continue;
		printf("\t");
		for (i = c->min, s = c->name; *s; s++, i--) {
			if (i > 0)
				n = toupper(*s);
			else
				n = *s;
			putchar(n);
		}
		if (*c->args)
			printf (" %s", c->args);
		printf ("\n");
	}
}

void usage ()
{
	fprintf (stderr, "usage: wfdcontrol [-vs] [-f disk] [command args ...]\n");
	exit (1);
}

int main (int argc, char **argv)
{
	int cmd;
	char *arg;

	for (;;) {
		switch (getopt (argc, argv, "svhf:")) {
		case EOF:
			break;
		case 's':
			verbose = 0;
			continue;
		case 'v':
			verbose = 2;
			continue;
		case 'f':
			wfdname = optarg;
			continue;
		case 'h':
		default:
			usage ();
		}
		break;
	}
	argc -= optind;
	argv += optind;

	if (argc > 0 && ! strcasecmp (*argv, "help"))
		usage ();

	if (! wfdname) {
		wfdname = DEFAULT_WFD_DRIVE;
		warnx("no WFD device name specified, defaulting to %s", wfdname);
	}

	if (argc > 0) {
		char buf[80], *p;
		int len;

		for (p=buf; argc-->0; ++argv) {
			len = strlen (*argv);

			if (p + len >= buf + sizeof (buf) - 1)
				usage ();

			if (p > buf)
				*p++ = ' ';

			strcpy (p, *argv);
			p += len;
		}
		*p = 0;
		arg = parse (buf, &cmd);
		return (run (cmd, arg));
	}

	if (verbose == 1)
		verbose = isatty (0);

	if (verbose) {
		printf ("ATAPI LS-120/ZIP Control utility, version %s\n", VERSION);
		printf ("Type `?' for command list\n\n");
	}

	for (;;) {
		arg = input (&cmd);
		if (run (cmd, arg) < 0) {
			if (verbose)
				warn(NULL);
			close (fd);
			fd = -1;
		}
		fflush (stdout);
	}
}

int run (int cmd, char *arg)
{
	int rc;
	struct wfd_protect protect;
	char *password;

	switch (cmd) {

	case CMD_QUIT:
		exit (0);

	case CMD_STATUS:
		if (fd < 0 && ! open_wfd ())
			return (0);

		return status (arg);

	case CMD_RESUME:
		if (fd < 0 && ! open_wfd ())
			return (0);

		return ioctl (fd, CDIOCRESUME);

	case CMD_RESET:
		if (fd < 0 && ! open_wfd ())
			return (0);

		rc = ioctl (fd, CDIOCRESET);
		if (rc < 0)
			return rc;
		close(fd);
		fd = -1;
		return (0);

	case CMD_DEBUG:
		if (fd < 0 && ! open_wfd ())
			return (0);

		if (! strcasecmp (arg, "on"))
			return ioctl (fd, CDIOCSETDEBUG);

		if (! strcasecmp (arg, "off"))
			return ioctl (fd, CDIOCCLRDEBUG);

		warnx("invalid command arguments");

		return (0);

	case CMD_EJECT:
		if (fd < 0 && ! open_wfd ())
			return (0);

		(void) ioctl (fd, CDIOCALLOW);
		rc = ioctl (fd, CDIOCEJECT);
		if (rc < 0)
			return (rc);
		return (0);

	case CMD_PROTECT:
		if (fd < 0 && ! open_wfd ())
			return (0);

		if (! strcasecmp (arg, "on")) {
			printf("Type password. Type ENTER if you don't need password.\n");
		}
		if (! strcasecmp (arg, "on") || ! strcasecmp (arg, "rwon")) {
			password = getpass("Password:");
			if (*password == '\0') {
				protect.protect_mode = WP;
				protect.passwd[0] = '\0';
			} else {
				strcpy(protect.passwd, password);
			  	password = getpass("Retype password:");
				if (strcmp(protect.passwd, password)) {
					warnx("Mismatch; try again.");
					return 0;
				}
				protect.protect_mode = PWP;
			}
		} else if (! strcasecmp (arg, "off")) {
			printf("Type password. Type ENTER if password is not needed.\n");
			password = getpass("Password:");
			strcpy(protect.passwd, password);
			protect.protect_mode = UNLOCK;
		} else
			warnx("invalid command arguments");

		protect.passwd_length = strlen(protect.passwd);
		if (protect.passwd_length > 32)
			protect.passwd_length = 32;
		ioctl (fd, WFD_PROTECT, &protect);
		return (0);

	case CMD_FORMAT:
		if (fd < 0 && ! open_wfd ())
			return (0);

		ioctl (fd, WFD_FORMAT);
		return (0);
	default:
	case CMD_HELP:
		help ();
		return (0);

	}
}

int status (char *arg)
{
	struct wfd_status status;

	bzero (&status, sizeof (status));
	ioctl (fd, WFD_STATUS);

	printf("Drive type: ");
	if (status.drive_type == LS120) {
		printf("LS-120\n");
		switch (status.medium_type) {
		case MDT_UNKNOWN:
			printf ("medium type unknown (no disk)\n");
			break;
		case MDT_2DD_UN:
			printf ("2DD(capacity unknown) floppy disk loaded\n");
			break;
		case MDT_2DD:
			printf ("720KB floppy disk loaded\n");
			break;
		case MDT_2HD_UN:
			printf ("2HD(capacity unknown) floppy disk loaded\n");
			break;
		case MDT_2HD_12_98:
			printf ("1.25MB(PC-9801 format) floppy disk loaded\n");
			break;
		case MDT_2HD_12:
			printf ("1.2MB floppy disk loaded\n");
			break;
		case MDT_2HD_144:
			printf ("1.44MB floppy disk loaded\n");
			break;
		case MDT_LS120:
			printf ("120MB floppy disk loaded\n");
			break;
		default:
			printf ("medium type=0x%x\n", status.medium_type);
			break;
		}
	} else {
		printf("ZIP\n");
		switch (status.disk_status) {
		case ZDS_SPIN_SPEED:
			printf ("Disk present and spinning at speed.\n");
			break;
		case ZDS_SPIN_UP:
			printf ("Disk present and spinning up.\n");
			break;
		case ZDS_NO_DISK:
			printf ("Disk not present.\n");
			break;
		case ZDS_AUTO_DOWN:
			printf ("Disk present and auto-spun down.\n");
			break;
		case ZDS_MOTOR_STALL:
			printf ("Moter stalled.\n");
			break;
		case ZDS_UNKNOWN_STATUS:
		default:
			printf ("Disk status unknown.\n");
			break;
		}
	}
	printf ("%u cyls, %u heads, %u S/T, %u B/S\n",
		status.cyls, status.heads, status.sectors, status.sector_size);

	printf ("Motor delay %u, %d rpm\n",
		status.motor_delay, status.rpm);

	if (status.drive_type == ZIP) {
		printf("Protect mode: ");
		switch (status.protect_mode) {
		case UNLOCK:
			printf("Unlock Mode\n");
			break;
		case WP:
			printf("Write Protect Mode\n");
			break;
		case PWP:
			printf("Password Write Protect Mode\n");
			break;
		case RWP:
			printf("Read/Write Protect Mode\n");
			break;
		case TUNLOCK:
			printf("Temporary Unlock Mode\n");
			break;
		default:
			printf("Unknown %x\n", status.protect_mode);
			break;
		}
		if (status.prev_status)
			printf ("Drive is in the Prevent Media Removal mode.\n");
		else
			printf ("Drive is in the Allow Media Removal mode.\n");
		printf("Last LBA: %lu\n", status.last_lba);
		printf("Spare Blocks Available for Reassignment - Side0: %u\n", status.space_s0);
		printf("Number of Blocks Reassigned - Side0: %u\n", status.number_s0);
		printf("Spare Blocks Available for Reassignment - Side1: %u\n", status.space_s1);
		printf("Number of Blocks Reassigned - Side1: %u\n", status.number_s1);
	}
	return 0;
}

char *input (int *cmd)
{
	static char buf[80];
	char *p;

	do {
		if (verbose)
			fprintf (stderr, "cdcontrol> ");
		if (! fgets (buf, sizeof (buf), stdin)) {
			*cmd = CMD_QUIT;
			fprintf (stderr, "\r\n");
			return (0);
		}
		p = parse (buf, cmd);
	} while (! p);
	return (p);
}

char *parse (char *buf, int *cmd)
{
	struct cmdtab *c;
	char *p;
	int len;

	for (p=buf; isspace (*p); p++)
		continue;

	for (buf = p; *p && ! isspace (*p); p++)
		continue;
  
	len = p - buf;
	if (! len)
		return (0);

	if (*p) {                       /* It must be a spacing character! */
		char *q;

		*p++ = 0;
		for (q=p; *q && *q != '\n' && *q != '\r'; q++)
			continue;
		*q = 0;
	}

	*cmd = -1;
	for (c=cmdtab; c->name; ++c) {
		/* Is it an exact match? */
		if (! strcasecmp (buf, c->name)) {
  			*cmd = c->command;
  			break;
  		}

		/* Try short hand forms then... */
		if (len >= c->min && ! strncasecmp (buf, c->name, len)) {
			if (*cmd != -1 && *cmd != c->command) {
				warnx("ambiguous command");
				return (0);
			}
			*cmd = c->command;
  		}
	}

	if (*cmd == -1) {
		warnx("invalid command, enter ``help'' for commands");
		return (0);
	}

	while (isspace (*p))
		p++;
	return p;
}

int open_wfd ()
{
	char devbuf[80];

	if (fd > -1)
		return (1);

	if (*wfdname == '/')
		strcpy (devbuf, wfdname);
	else if (*wfdname == 'r')
		sprintf (devbuf, "/dev/%s", wfdname);
	else
		sprintf (devbuf, "/dev/r%s", wfdname);

	fd = open (devbuf, O_RDONLY);

	if (fd < 0 && errno == ENOENT) {
		strcat (devbuf, DEFAULT_WFD_PARTITION);
		fd = open (devbuf, O_RDONLY);
	}

	if (fd < 0) {
		if (errno == ENXIO) {
			/*  ENXIO has an overloaded meaning here.
			 *  The original "Device not configured" should
			 *  be interpreted as "No disk in drive %s". */
			warnx("no disk in drive %s", devbuf);
			return (0);
		}
		err(1, "%s", devbuf);
	}
	return (1);
}
